<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\PromoCode;
use App\Models\Bookmaker;

class PromoCodeSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $bookmakers = Bookmaker::all();

        foreach ($bookmakers as $bookmaker) {
            $promoCodesCount = rand(2, 4);
            
            for ($i = 0; $i < $promoCodesCount; $i++) {
                $this->createPromoCode($bookmaker, $i + 1);
            }
        }
    }

    private function createPromoCode($bookmaker, $index)
    {
        $bonusTypes = ['welcome', 'deposit', 'free_bet', 'cashback', 'reload'];
        $bonusType = $bonusTypes[array_rand($bonusTypes)];
        
        $titles = [
            'welcome' => ['Приветственный бонус', 'Бонус новичка', 'Добро пожаловать'],
            'deposit' => ['Бонус на депозит', 'Пополнение с бонусом', 'Депозит + бонус'],
            'free_bet' => ['Фрибет', 'Бесплатная ставка', 'Свободная ставка'],
            'cashback' => ['Кэшбэк', 'Возврат средств', 'Компенсация'],
            'reload' => ['Бонус на пополнение', 'Релоад бонус', 'Повторный бонус']
        ];

        $title = $titles[$bonusType][array_rand($titles[$bonusType])];
        
        $bonusAmount = $this->getBonusAmount($bonusType, $bookmaker->currency);
        $minDeposit = $this->getMinDeposit($bonusType, $bookmaker->currency);
        
        PromoCode::create([
            'bookmaker_id' => $bookmaker->id,
            'code' => strtoupper(substr($bookmaker->slug, 0, 3) . rand(1000, 9999)),
            'title' => $title . ' ' . $index,
            'description' => $this->getDescription($bonusType, $bonusAmount, $minDeposit, $bookmaker->currency),
            'bonus_type' => $bonusType,
            'bonus_amount' => $bonusAmount,
            'bonus_currency' => $bookmaker->currency,
            'min_deposit' => $minDeposit,
            'max_bonus' => $bonusAmount * 1.5,
            'wagering_requirement' => rand(3, 10),
            'valid_from' => now(),
            'valid_until' => now()->addDays(rand(30, 90)),
            'max_uses' => rand(100, 1000),
            'current_uses' => rand(0, 50),
            'is_active' => true,
            'is_featured' => rand(0, 1),
            'points_reward' => rand(10, 100),
        ]);
    }

    private function getBonusAmount($bonusType, $currency)
    {
        $multipliers = [
            'RUB' => 1,
            'USD' => 100,
            'EUR' => 90,
            'GBP' => 80,
            'BTC' => 0.001
        ];

        $baseAmounts = [
            'welcome' => rand(1000, 5000),
            'deposit' => rand(500, 2000),
            'free_bet' => rand(200, 1000),
            'cashback' => rand(5, 20),
            'reload' => rand(300, 1500)
        ];

        $baseAmount = $baseAmounts[$bonusType];
        $multiplier = $multipliers[$currency] ?? 1;

        return $baseAmount * $multiplier;
    }

    private function getMinDeposit($bonusType, $currency)
    {
        $multipliers = [
            'RUB' => 1,
            'USD' => 100,
            'EUR' => 90,
            'GBP' => 80,
            'BTC' => 0.001
        ];

        $baseAmounts = [
            'welcome' => rand(100, 500),
            'deposit' => rand(200, 1000),
            'free_bet' => rand(100, 500),
            'cashback' => rand(500, 2000),
            'reload' => rand(300, 1000)
        ];

        $baseAmount = $baseAmounts[$bonusType];
        $multiplier = $multipliers[$currency] ?? 1;

        return $baseAmount * $multiplier;
    }

    private function getDescription($bonusType, $bonusAmount, $minDeposit, $currency)
    {
        $currencySymbols = [
            'RUB' => '₽',
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
            'BTC' => '₿'
        ];

        $symbol = $currencySymbols[$currency] ?? $currency;

        $descriptions = [
            'welcome' => "Получите {$symbol}" . number_format($bonusAmount, 0, ',', ' ') . " за регистрацию и первый депозит от {$symbol}" . number_format($minDeposit, 0, ',', ' ') . ". Вейджер: 3x.",
            'deposit' => "Бонус {$symbol}" . number_format($bonusAmount, 0, ',', ' ') . " на депозит от {$symbol}" . number_format($minDeposit, 0, ',', ' ') . ". Вейджер: 5x.",
            'free_bet' => "Фрибет на {$symbol}" . number_format($bonusAmount, 0, ',', ' ') . " при депозите от {$symbol}" . number_format($minDeposit, 0, ',', ' ') . ". Вейджер: 3x.",
            'cashback' => "Кэшбэк {$bonusAmount}% с депозита от {$symbol}" . number_format($minDeposit, 0, ',', ' ') . ". Вейджер: 7x.",
            'reload' => "Бонус {$symbol}" . number_format($bonusAmount, 0, ',', ' ') . " на повторное пополнение от {$symbol}" . number_format($minDeposit, 0, ',', ' ') . ". Вейджер: 4x."
        ];

        return $descriptions[$bonusType] ?? "Специальное предложение от букмекера.";
    }
}
