<?php

namespace App\Services;

use App\Models\ParserJob;
use App\Models\ParsedPost;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Symfony\Component\DomCrawler\Crawler;


class ParserService
{
    protected $client;
    protected $job;

    public function __construct()
    {
        $this->client = new Client([
            'timeout' => 30,
            'headers' => [
                'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'
            ]
        ]);
    }

    /**
     * Запуск парсинга
     */
    public function parse(ParserJob $job)
    {
        $this->job = $job;
        $job->update(['status' => 'running']);

        try {
            $totalPosts = 0;
            $parsedPosts = 0;

            // Проходим по всем страницам
            for ($page = $job->start_page; $page <= $job->end_page; $page++) {
                $url = $this->buildPageUrl($page);
                
                try {
                    $response = $this->client->get($url);
                    $html = $response->getBody()->getContents();
                    
                    $crawler = new Crawler($html);
                    $posts = $crawler->filter($job->post_selector);
                    
                    $totalPosts += $posts->count();
                    
                    // Парсим каждый пост
                    $posts->each(function (Crawler $postNode) use ($page, &$parsedPosts) {
                        $this->parsePost($postNode, $page);
                        $parsedPosts++;
                    });
                    
                    // Обновляем прогресс
                    $this->job->update([
                        'total_posts' => $totalPosts,
                        'parsed_posts' => $parsedPosts
                    ]);
                    
                    // Небольшая задержка между запросами
                    sleep(1);
                    
                } catch (RequestException $e) {
                    \Log::error("Ошибка при парсинге страницы {$page}: " . $e->getMessage());
                    continue;
                }
            }
            
            $job->update(['status' => 'completed']);
            
        } catch (\Exception $e) {
            $job->update([
                'status' => 'failed',
                'error_message' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Построение URL для страницы
     */
    protected function buildPageUrl(int $page): string
    {
        if ($this->job->pagination_pattern) {
            $pattern = str_replace('{page}', $page, $this->job->pagination_pattern);
            return $this->job->base_url . $pattern;
        }
        
        return $this->job->base_url;
    }

    /**
     * Парсинг отдельного поста
     */
    protected function parsePost(Crawler $postNode, int $pageNumber)
    {
        try {
            // Извлекаем заголовок
            $title = $this->extractText($postNode, $this->job->title_selector);
            
            // Извлекаем контент
            $content = $this->extractText($postNode, $this->job->content_selector);
            
            // Извлекаем ссылку
            $link = $this->extractLink($postNode, $this->job->link_selector);
            
            // Извлекаем и сохраняем изображение
            $imagePath = null;
            $imageUrl = null;
            
            if ($this->job->image_selector) {
                $imageUrl = $this->extractImage($postNode, $this->job->image_selector);
                if ($imageUrl) {
                    $imagePath = $this->saveImage($imageUrl);
                }
            }
            
            // Сохраняем пост в базу
            ParsedPost::create([
                'parser_job_id' => $this->job->id,
                'title' => $title,
                'content' => $content,
                'original_url' => $link,
                'image_path' => $imagePath,
                'image_url' => $imageUrl,
                'page_number' => $pageNumber,
                'metadata' => [
                    'parsed_at' => now()->toISOString(),
                    'user_agent' => $this->client->getConfig('headers')['User-Agent']
                ]
            ]);
            
        } catch (\Exception $e) {
            \Log::error("Ошибка при парсинге поста: " . $e->getMessage());
        }
    }

    /**
     * Извлечение текста по селектору
     */
    protected function extractText(Crawler $node, string $selector): string
    {
        try {
            $element = $node->filter($selector)->first();
            return trim($element->text());
        } catch (\Exception $e) {
            return '';
        }
    }

    /**
     * Извлечение ссылки по селектору
     */
    protected function extractLink(Crawler $node, ?string $selector): ?string
    {
        if (!$selector) return null;
        
        try {
            $element = $node->filter($selector)->first();
            $href = $element->attr('href');
            
            if ($href && !Str::startsWith($href, ['http://', 'https://'])) {
                // Преобразуем относительную ссылку в абсолютную
                $baseUrl = parse_url($this->job->base_url, PHP_URL_SCHEME) . '://' . parse_url($this->job->base_url, PHP_URL_HOST);
                $href = $baseUrl . $href;
            }
            
            return $href;
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Извлечение изображения по селектору
     */
    protected function extractImage(Crawler $node, string $selector): ?string
    {
        try {
            $element = $node->filter($selector)->first();
            $src = $element->attr('src') ?: $element->attr('data-src');
            
            if ($src && !Str::startsWith($src, ['http://', 'https://'])) {
                // Преобразуем относительную ссылку в абсолютную
                $baseUrl = parse_url($this->job->base_url, PHP_URL_SCHEME) . '://' . parse_url($this->job->base_url, PHP_URL_HOST);
                $src = $baseUrl . $src;
            }
            
            return $src;
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Сохранение изображения
     */
    protected function saveImage(string $imageUrl): ?string
    {
        try {
            $response = $this->client->get($imageUrl);
            $imageData = $response->getBody()->getContents();
            
            // Генерируем уникальное имя файла
            $extension = $this->getImageExtension($imageData);
            $filename = 'parsed_images/' . Str::uuid() . '.' . $extension;
            
            // Сохраняем изображение
            Storage::disk('public')->put($filename, $imageData);
            
            return $filename;
            
        } catch (\Exception $e) {
            \Log::error("Ошибка при сохранении изображения {$imageUrl}: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Определение расширения изображения
     */
    protected function getImageExtension(string $imageData): string
    {
        $finfo = new \finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->buffer($imageData);
        
        $extensions = [
            'image/jpeg' => 'jpg',
            'image/jpg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp'
        ];
        
        return $extensions[$mimeType] ?? 'jpg';
    }

    /**
     * Тестирование селекторов
     */
    public function testSelectors(string $url, array $selectors): array
    {
        try {
            $response = $this->client->get($url);
            $html = $response->getBody()->getContents();
            $crawler = new Crawler($html);
            
            $results = [];
            
            foreach ($selectors as $name => $selector) {
                try {
                    $elements = $crawler->filter($selector);
                    $results[$name] = [
                        'count' => $elements->count(),
                        'sample' => $elements->count() > 0 ? trim($elements->first()->text()) : null
                    ];
                } catch (\Exception $e) {
                    $results[$name] = [
                        'count' => 0,
                        'error' => $e->getMessage()
                    ];
                }
            }
            
            return $results;
            
        } catch (\Exception $e) {
            return ['error' => $e->getMessage()];
        }
    }
} 