<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Laravel\Sanctum\HasApiTokens;
use App\Models\UserFollow;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'role',
        'username',
        'avatar',
        'bio',
        'phone',
        'birth_date',
        'gender',
        'country',
        'city',
        'referral_code',
        'referred_by',
        'total_points',
        'available_points',
        'current_points',
        'total_profit',
        'predictions_count',
        'correct_predictions',
        'success_rate',
        'is_verified',
        'last_login_at',
        'is_online',
        'last_activity_at',
        'preferences',
        'rank_id',
        'google_id',
        'email_verified_at',
    ];

    /**
     * The attributes that aren't mass assignable.
     *
     * @var array
     */
    protected $guarded = [
        'avatar_url', // Виртуальное поле, не должно быть массово назначаемым
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'avatar_url',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'birth_date' => 'date',
            'last_login_at' => 'datetime',
            'last_activity_at' => 'datetime',
            'is_online' => 'boolean',
            'preferences' => 'array',
            'success_rate' => 'decimal:2',
            'total_profit' => 'decimal:2',
        ];
    }

    /**
     * Get the user's rank.
     */
    public function rank(): BelongsTo
    {
        return $this->belongsTo(Rank::class);
    }

    /**
     * Get the user's current rank based on points (always calculate by points)
     */
    public function getCurrentRankAttribute()
    {
        // Всегда определяем ранг по total_points (общее количество очков)
        return Rank::getRankByPoints($this->total_points ?? 0);
    }

    /**
     * Get the user who referred this user.
     */
    public function referrer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'referred_by');
    }

    /**
     * Get the users referred by this user.
     */
    public function referrals(): HasMany
    {
        return $this->hasMany(User::class, 'referred_by');
    }

    /**
     * Get the user's predictions.
     */
    public function predictions(): HasMany
    {
        return $this->hasMany(Prediction::class);
    }

    /**
     * Get the user's points history.
     */
    public function pointsHistory(): HasMany
    {
        return $this->hasMany(UserPoints::class);
    }

    /**
     * Get the user's promo code usages.
     */
    public function promoCodeUsages(): HasMany
    {
        return $this->hasMany(PromoCodeUsage::class);
    }

    /**
     * Get users that this user follows.
     */
    public function follows(): HasMany
    {
        return $this->hasMany(UserFollow::class, 'follower_id');
    }

    /**
     * Get users that follow this user.
     */
    public function followers(): HasMany
    {
        return $this->hasMany(UserFollow::class, 'followed_id');
    }

    /**
     * Get the user's stories.
     */
    public function stories(): HasMany
    {
        return $this->hasMany(Story::class);
    }

    /**
     * Get the user's story views.
     */
    public function storyViews(): HasMany
    {
        return $this->hasMany(StoryView::class);
    }

    /**
     * Get the user's story likes.
     */
    public function storyLikes(): HasMany
    {
        return $this->hasMany(StoryLike::class);
    }

    /**
     * Check if user is admin.
     */
    public function isAdmin(): bool
    {
        return $this->role === 'admin';
    }

    /**
     * Check if user is moderator.
     */
    public function isModerator(): bool
    {
        return $this->role === 'moderator';
    }

    /**
     * Generate unique referral code.
     */
    public static function generateReferralCode(): string
    {
        do {
            $code = strtoupper(substr(md5(uniqid()), 0, 8));
        } while (self::where('referral_code', $code)->exists());

        return $code;
    }

    /**
     * Get user initials for avatar fallback.
     */
    public function getInitials(): string
    {
        $name = trim($this->name);
        $words = explode(' ', $name);
        
        if (count($words) >= 2) {
            return strtoupper(substr($words[0], 0, 1) . substr($words[1], 0, 1));
        }
        
        return strtoupper(substr($name, 0, 2));
    }

    /**
     * Get avatar URL or null if not set.
     */
    public function getAvatarUrl(): ?string
    {
        if (!$this->avatar) {
            return null;
        }
        
        return asset('storage/' . $this->avatar);
    }

    /**
     * Update user's online status
     */
    public function updateOnlineStatus(bool $isOnline = true): void
    {
        $this->update([
            'is_online' => $isOnline,
            'last_activity_at' => $isOnline ? now() : null
        ]);
    }

    /**
     * Check if user is currently online (active in last 5 minutes)
     */
    public function isCurrentlyOnline(): bool
    {
        if (!$this->is_online) {
            return false;
        }

        // Считаем пользователя онлайн, если он был активен в последние 5 минут
        return $this->last_activity_at && $this->last_activity_at->diffInMinutes(now()) <= 5;
    }

    /**
     * Get online status text
     */
    public function getOnlineStatusText(): string
    {
        if ($this->isCurrentlyOnline()) {
            return 'В сети';
        }

        if ($this->last_activity_at) {
            $diff = $this->last_activity_at->diffForHumans();
            
            // Переводим английский текст на русский
            $translations = [
                'ago' => 'назад',
                'from now' => 'спустя',
                'second' => 'секунду',
                'seconds' => 'секунд',
                'minute' => 'минуту',
                'minutes' => 'минут',
                'hour' => 'час',
                'hours' => 'часов',
                'day' => 'день',
                'days' => 'дней',
                'week' => 'неделю',
                'weeks' => 'недель',
                'month' => 'месяц',
                'months' => 'месяцев',
                'year' => 'год',
                'years' => 'лет',
                'a' => '',
                'an' => '',
            ];
            
            $russianDiff = $diff;
            foreach ($translations as $english => $russian) {
                $russianDiff = str_replace($english, $russian, $russianDiff);
            }
            
            return "Был(а) в сети {$russianDiff}";
        }

        return 'Не в сети';
    }

    /**
     * Get online status class for CSS
     */
    public function getOnlineStatusClass(): string
    {
        if ($this->isCurrentlyOnline()) {
            return 'text-green-600';
        }

        if ($this->last_activity_at && $this->last_activity_at->diffInHours(now()) <= 24) {
            return 'text-yellow-600';
        }

        return 'text-gray-500';
    }

    /**
     * Get gender display text
     */
    public function getGenderText(): string
    {
        return match($this->gender) {
            'male' => 'Мужской',
            'female' => 'Женский',
            'other' => 'Другой',
            default => 'Не указан'
        };
    }

    /**
     * Get gender icon
     */
    public function getGenderIcon(): string
    {
        return match($this->gender) {
            'male' => '👨',
            'female' => '👩',
            'other' => '👤',
            default => '👤'
        };
    }

    /**
     * Check if user is male
     */
    public function isMale(): bool
    {
        return $this->gender === 'male';
    }

    /**
     * Check if user is female
     */
    public function isFemale(): bool
    {
        return $this->gender === 'female';
    }

    /**
     * Check if user has specified gender
     */
    public function hasGender(): bool
    {
        return !empty($this->gender);
    }

    /**
     * Обновить ранг пользователя на основе очков
     */
    public function updateRank(): void
    {
        $newRank = Rank::getRankByPoints($this->total_points);
        
        if ($newRank && $this->rank_id !== $newRank->id) {
            $this->rank_id = $newRank->id;
            $this->save();
        }
    }

    /**
     * Добавить очки пользователю
     */
    public function addPoints(int $points): void
    {
        $this->current_points += $points;
        $this->total_points += $points;
        $this->available_points += $points;
        $this->save();
        
        // Обновляем ранг
        $this->updateRank();
    }

    /**
     * Получить прогресс до следующего ранга
     */
    public function getRankProgress(): float
    {
        if (!$this->rank) {
            return 0.0;
        }

        return $this->rank->getProgressToNextRank($this->total_points);
    }

    /**
     * Получить очки до следующего ранга
     */
    public function getPointsToNextRank(): int
    {
        if (!$this->rank || $this->rank->isMaxRank()) {
            return 0;
        }

        $nextRank = $this->rank->getNextRank();
        return $nextRank->min_points - $this->total_points;
    }

    /**
     * Проверить, достиг ли пользователь максимального ранга
     */
    public function isMaxRank(): bool
    {
        return $this->rank ? $this->rank->isMaxRank() : false;
    }

    /**
     * Пользователи, которых заблокировал данный пользователь
     */
    public function blockedUsers(): HasMany
    {
        return $this->hasMany(UserBlacklist::class);
    }

    /**
     * Пользователи, которые заблокировали данного пользователя
     */
    public function blockedByUsers(): HasMany
    {
        return $this->hasMany(UserBlacklist::class, 'blocked_user_id');
    }

    /**
     * Заблокировать пользователя
     */
    public function blockUser(int $blockedUserId, ?string $reason = null): bool
    {
        // Нельзя заблокировать самого себя
        if ($this->id === $blockedUserId) {
            return false;
        }

        // Проверяем, не заблокирован ли уже
        if ($this->hasBlockedUser($blockedUserId)) {
            return false;
        }

        $blocked = UserBlacklist::create([
            'user_id' => $this->id,
            'blocked_user_id' => $blockedUserId,
            'reason' => $reason
        ]);

        if ($blocked) {
            // Удаляем взаимные подписки при блокировке
            UserFollow::where(function ($query) use ($blockedUserId) {
                $query->where('follower_id', $this->id)
                      ->where('followed_id', $blockedUserId);
            })->orWhere(function ($query) use ($blockedUserId) {
                $query->where('follower_id', $blockedUserId)
                      ->where('followed_id', $this->id);
            })->delete();
        }

        return $blocked !== null;
    }

    /**
     * Разблокировать пользователя
     */
    public function unblockUser(int $blockedUserId): bool
    {
        return UserBlacklist::where('user_id', $this->id)
                           ->where('blocked_user_id', $blockedUserId)
                           ->delete() > 0;
    }

    /**
     * Проверить, заблокирован ли пользователь
     */
    public function hasBlockedUser(int $blockedUserId): bool
    {
        return UserBlacklist::isBlocked($this->id, $blockedUserId);
    }

    /**
     * Проверить, заблокирован ли данный пользователь другим пользователем
     */
    public function isBlockedBy(int $userId): bool
    {
        return UserBlacklist::isBlocked($userId, $this->id);
    }

    /**
     * Получить список заблокированных пользователей
     */
    public function getBlockedUsers()
    {
        return UserBlacklist::getBlockedUsers($this->id);
    }

    /**
     * Получить список пользователей, которые заблокировали данного пользователя
     */
    public function getBlockedByUsers()
    {
        return UserBlacklist::getBlockedByUsers($this->id);
    }
}
