<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Carbon\Carbon;

class Story extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'content',
        'media_url',
        'type',
        'is_public',
        'expires_at'
    ];

    protected $casts = [
        'is_public' => 'boolean',
        'expires_at' => 'datetime',
    ];

    protected $appends = [
        'is_expired',
        'views_count',
        'likes_count',
        'is_liked_by_current_user',
        'is_viewed_by_current_user'
    ];

    /**
     * Get the user who created the story.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the story views.
     */
    public function views(): HasMany
    {
        return $this->hasMany(StoryView::class);
    }

    /**
     * Get the story likes.
     */
    public function likes(): HasMany
    {
        return $this->hasMany(StoryLike::class);
    }

    /**
     * Check if story is expired (older than 24 hours).
     */
    public function getIsExpiredAttribute(): bool
    {
        return $this->created_at->diffInHours(now()) >= 24;
    }

    /**
     * Get views count.
     */
    public function getViewsCountAttribute(): int
    {
        return $this->views()->count();
    }

    /**
     * Get likes count.
     */
    public function getLikesCountAttribute(): int
    {
        return $this->likes()->count();
    }

    /**
     * Check if story is liked by current user.
     */
    public function getIsLikedByCurrentUserAttribute(): bool
    {
        if (!auth()->check()) {
            return false;
        }
        return $this->likes()->where('user_id', auth()->id())->exists();
    }

    /**
     * Check if story is viewed by current user.
     */
    public function getIsViewedByCurrentUserAttribute(): bool
    {
        if (!auth()->check()) {
            return false;
        }
        return $this->views()->where('user_id', auth()->id())->exists();
    }

    /**
     * Scope to get only non-expired stories.
     */
    public function scopeNotExpired($query)
    {
        return $query->where('created_at', '>=', now()->subHours(24));
    }

    /**
     * Scope to get only public stories.
     */
    public function scopePublic($query)
    {
        return $query->where('is_public', true);
    }

    /**
     * Scope to get stories by user with minimum points.
     */
    public function scopeByQualifiedUsers($query, int $minPoints = 3000)
    {
        return $query->whereHas('user', function ($q) use ($minPoints) {
            $q->where('total_points', '>=', $minPoints);
        });
    }

    /**
     * Boot method to set expires_at when creating.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($story) {
            $story->expires_at = now()->addHours(24);
        });
    }
}
