<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Slot extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'provider',
        'theme',
        'rtp',
        'volatility',
        'min_bet',
        'max_bet',
        'max_win',
        'paylines',
        'reels',
        'release_date',
        'rating',
        'plays_count',
        'features',
        'bonus_features',
        'demo_url',
        'real_url',
        'iframe_url',
        'image',
        'is_featured',
        'is_active',
        'meta_title',
        'meta_description',
        'meta_keywords'
    ];

    protected $casts = [
        'rtp' => 'float',
        'volatility' => 'float',
        'min_bet' => 'integer',
        'max_bet' => 'integer',
        'max_win' => 'integer',
        'paylines' => 'integer',
        'reels' => 'integer',
        'rating' => 'float',
        'plays_count' => 'integer',
        'is_featured' => 'boolean',
        'is_active' => 'boolean',
        'features' => 'array',
        'bonus_features' => 'array'
    ];

    public function getImageUrlAttribute()
    {
        if ($this->image) {
            return asset('storage/' . $this->image);
        }
        return asset('images/default-slot-image.png');
    }

    public function getFormattedRatingAttribute()
    {
        return number_format($this->rating, 1);
    }

    public function getFormattedRtpAttribute()
    {
        return number_format($this->rtp, 2) . '%';
    }

    public function getFormattedVolatilityAttribute()
    {
        if (!$this->volatility) return 'Неизвестно';
        
        if ($this->volatility <= 2.0) return 'Низкая';
        if ($this->volatility <= 4.0) return 'Средняя';
        return 'Высокая';
    }

    public function getFormattedMinBetAttribute()
    {
        return number_format($this->min_bet, 0) . ' ₽';
    }

    public function getFormattedMaxBetAttribute()
    {
        return number_format($this->max_bet, 0) . ' ₽';
    }

    public function getFormattedMaxWinAttribute()
    {
        return number_format($this->max_win, 0) . ' ₽';
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    public function scopeTopRated($query)
    {
        return $query->orderBy('rating', 'desc');
    }

    public function scopePopular($query)
    {
        return $query->orderBy('plays_count', 'desc');
    }

    public function scopeByProvider($query, $provider)
    {
        return $query->where('provider', $provider);
    }

    public function scopeByTheme($query, $theme)
    {
        return $query->where('theme', $theme);
    }

    public function getRouteKeyName()
    {
        return 'slug';
    }
}
