<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Rank extends Model
{
    protected $fillable = [
        'name',
        'slug',
        'min_points',
        'max_points',
        'color',
        'icon',
        'description',
        'level'
    ];

    protected $casts = [
        'min_points' => 'integer',
        'max_points' => 'integer',
        'level' => 'integer'
    ];

    /**
     * Пользователи с этим рангом
     */
    public function users(): HasMany
    {
        return $this->hasMany(User::class);
    }

    /**
     * Получить ранг по количеству очков
     */
    public static function getRankByPoints(int $points): ?self
    {
        // Если очков больше максимального ранга, возвращаем максимальный ранг
        $maxRank = self::where('level', 10)->first();
        if ($maxRank && $points >= $maxRank->min_points) {
            return $maxRank;
        }
        
        // Иначе ищем ранг по диапазону
        return self::where('min_points', '<=', $points)
            ->where('max_points', '>=', $points)
            ->first();
    }

    /**
     * Получить следующий ранг
     */
    public function getNextRank(): ?self
    {
        return self::where('level', '>', $this->level)
            ->orderBy('level')
            ->first();
    }

    /**
     * Получить предыдущий ранг
     */
    public function getPreviousRank(): ?self
    {
        return self::where('level', '<', $this->level)
            ->orderBy('level', 'desc')
            ->first();
    }

    /**
     * Проверить, является ли это максимальным рангом
     */
    public function isMaxRank(): bool
    {
        return $this->level === 10;
    }

    /**
     * Получить прогресс до следующего ранга
     */
    public function getProgressToNextRank(int $currentPoints): float
    {
        $nextRank = $this->getNextRank();
        
        if (!$nextRank) {
            return 100.0; // Максимальный ранг
        }

        $pointsInCurrentRank = $currentPoints - $this->min_points;
        $pointsNeededForNextRank = $nextRank->min_points - $this->min_points;
        
        return min(100.0, ($pointsInCurrentRank / $pointsNeededForNextRank) * 100);
    }
}
