<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;

class NewsComment extends Model
{
    use HasFactory;

    protected $fillable = [
        'news_id',
        'user_id',
        'parent_id',
        'content',
        'status',
        'is_edited',
        'edited_at',
        'likes_count',
        'replies_count',
    ];

    protected $casts = [
        'is_edited' => 'boolean',
        'edited_at' => 'datetime',
        'likes_count' => 'integer',
        'replies_count' => 'integer',
    ];

    /**
     * Новость, к которой относится комментарий
     */
    public function news(): BelongsTo
    {
        return $this->belongsTo(News::class);
    }

    /**
     * Автор комментария
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Родительский комментарий (для вложенных комментариев)
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(NewsComment::class, 'parent_id');
    }

    /**
     * Ответы на комментарий
     */
    public function replies(): HasMany
    {
        return $this->hasMany(NewsComment::class, 'parent_id');
    }

    /**
     * Все ответы (рекурсивно)
     */
    public function allReplies(): HasMany
    {
        return $this->hasMany(NewsComment::class, 'parent_id')
            ->with('replies');
    }

    /**
     * Лайки комментария
     */
    public function likes(): HasMany
    {
        return $this->hasMany(NewsCommentLike::class);
    }

    /**
     * Точки пользователя за комментарий
     */
    public function userPoints(): MorphMany
    {
        return $this->morphMany(UserPoints::class, 'pointable');
    }

    /**
     * Проверить, является ли комментарий ответом
     */
    public function isReply(): bool
    {
        return !is_null($this->parent_id);
    }

    /**
     * Проверить, имеет ли комментарий ответы
     */
    public function hasReplies(): bool
    {
        return $this->replies_count > 0;
    }

    /**
     * Получить уровень вложенности комментария
     */
    public function getDepthAttribute(): int
    {
        $depth = 0;
        $parent = $this->parent;
        
        while ($parent) {
            $depth++;
            $parent = $parent->parent;
        }
        
        return $depth;
    }

    /**
     * Проверить, можно ли ответить на комментарий (максимум 3 уровня вложенности)
     */
    public function canReply(): bool
    {
        return $this->depth < 3;
    }

    /**
     * Проверить, лайкнул ли пользователь комментарий
     */
    public function isLikedBy(User $user): bool
    {
        return $this->likes()->where('user_id', $user->id)->where('type', 'like')->exists();
    }

    /**
     * Scope для одобренных комментариев
     */
    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    /**
     * Scope для комментариев без родителя (корневые комментарии)
     */
    public function scopeRoot($query)
    {
        return $query->whereNull('parent_id');
    }

    /**
     * Scope для ответов
     */
    public function scopeReplies($query)
    {
        return $query->whereNotNull('parent_id');
    }

    /**
     * Scope для сортировки по дате создания
     */
    public function scopeLatest($query)
    {
        return $query->orderBy('created_at', 'desc');
    }

    /**
     * Scope для сортировки по популярности
     */
    public function scopePopular($query)
    {
        return $query->orderBy('likes_count', 'desc')
            ->orderBy('created_at', 'desc');
    }
}
