<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Str;

class News extends Model
{
    use HasFactory;

    protected $fillable = [
        'category_id',
        'author_id',
        'title',
        'slug',
        'excerpt',
        'content',
        'featured_image',
        'images',
        'tags',
        'status',
        'is_featured',
        'is_pinned',
        'published_at',
        'views_count',
        'likes_count',
        'comments_count',
        'meta_title',
        'meta_description',
        'meta_keywords',
    ];

    protected $casts = [
        'images' => 'array',
        'tags' => 'array',
        'meta_keywords' => 'array',
        'is_featured' => 'boolean',
        'is_pinned' => 'boolean',
        'published_at' => 'datetime',
        'views_count' => 'integer',
        'likes_count' => 'integer',
        'comments_count' => 'integer',
    ];

    /**
     * Boot method для автоматической генерации slug
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($news) {
            if (empty($news->slug)) {
                $news->slug = Str::slug($news->title);
            }
        });

        static::updating(function ($news) {
            if ($news->isDirty('title') && empty($news->slug)) {
                $news->slug = Str::slug($news->title);
            }
        });
    }

    /**
     * Категория новости
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(NewsCategory::class, 'category_id');
    }

    /**
     * Автор новости
     */
    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'author_id');
    }

    /**
     * Комментарии к новости
     */
    public function comments(): HasMany
    {
        return $this->hasMany(NewsComment::class)->whereNull('parent_id');
    }

    /**
     * Все комментарии (включая ответы)
     */
    public function allComments(): HasMany
    {
        return $this->hasMany(NewsComment::class);
    }

    /**
     * Лайки новости
     */
    public function likes(): HasMany
    {
        return $this->hasMany(NewsLike::class);
    }

    /**
     * Просмотры новости
     */
    public function views(): HasMany
    {
        return $this->hasMany(NewsView::class);
    }

    /**
     * Точки пользователя (для системы рейтинга)
     */
    public function userPoints(): MorphMany
    {
        return $this->morphMany(UserPoints::class, 'pointable');
    }

    /**
     * Проверить, лайкнул ли пользователь новость
     */
    public function isLikedBy(User $user): bool
    {
        return $this->likes()->where('user_id', $user->id)->where('type', 'like')->exists();
    }

    /**
     * Проверить, дизлайкнул ли пользователь новость
     */
    public function isDislikedBy(User $user): bool
    {
        return $this->likes()->where('user_id', $user->id)->where('type', 'dislike')->exists();
    }

    /**
     * Проверить, просмотрел ли пользователь новость
     */
    public function isViewedBy(User $user): bool
    {
        return $this->views()->where('user_id', $user->id)->exists();
    }

    /**
     * Получить URL новости
     */
    public function getUrlAttribute(): string
    {
        return '/news/' . $this->slug;
    }

    /**
     * Получить краткое описание
     */
    public function getShortExcerptAttribute(): string
    {
        if ($this->excerpt) {
            return Str::limit($this->excerpt, 150);
        }
        return Str::limit(strip_tags($this->content), 150);
    }

    /**
     * Получить время чтения (примерно)
     */
    public function getReadingTimeAttribute(): int
    {
        $wordsPerMinute = 200;
        $wordCount = str_word_count(strip_tags($this->content));
        return max(1, round($wordCount / $wordsPerMinute));
    }

    /**
     * Scope для опубликованных новостей
     */
    public function scopePublished($query)
    {
        return $query->where('status', 'published')
            ->whereNotNull('published_at')
            ->where('published_at', '<=', now());
    }

    /**
     * Scope для выделенных новостей
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    /**
     * Scope для закрепленных новостей
     */
    public function scopePinned($query)
    {
        return $query->where('is_pinned', true);
    }

    /**
     * Scope для поиска по заголовку и содержанию
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('title', 'like', "%{$search}%")
              ->orWhere('content', 'like', "%{$search}%")
              ->orWhere('excerpt', 'like', "%{$search}%");
        });
    }

    /**
     * Scope для фильтрации по категории
     */
    public function scopeByCategory($query, $categoryId)
    {
        return $query->where('category_id', $categoryId);
    }

    /**
     * Scope для сортировки по популярности
     */
    public function scopePopular($query)
    {
        return $query->orderBy('views_count', 'desc')
            ->orderBy('likes_count', 'desc')
            ->orderBy('published_at', 'desc');
    }

    /**
     * Scope для сортировки по дате публикации
     */
    public function scopeLatest($query)
    {
        return $query->orderBy('published_at', 'desc');
    }
}
