<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Casino extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'logo',
        'website',
        'rating',
        'min_deposit',
        'max_deposit',
        'min_withdrawal',
        'max_withdrawal',
        'withdrawal_time',
        'license',
        'founded_year',
        'headquarters',
        'languages',
        'currencies',
        'payment_methods',
        'games_count',
        'providers_count',
        'is_featured',
        'is_active',
        'meta_title',
        'meta_description',
        'meta_keywords'
    ];

    protected $casts = [
        'rating' => 'float',
        'min_deposit' => 'float',
        'max_deposit' => 'float',
        'min_withdrawal' => 'float',
        'max_withdrawal' => 'float',
        'founded_year' => 'integer',
        'games_count' => 'integer',
        'providers_count' => 'integer',
        'is_featured' => 'boolean',
        'is_active' => 'boolean',
        'languages' => 'array',
        'currencies' => 'array',
        'payment_methods' => 'array'
    ];

    public function getLogoUrlAttribute()
    {
        if ($this->logo) {
            return asset('storage/' . $this->logo);
        }
        return asset('images/default-casino-logo.png');
    }

    public function getFormattedRatingAttribute()
    {
        return number_format($this->rating, 1);
    }

    public function getFormattedMinDepositAttribute()
    {
        return number_format($this->min_deposit, 0) . ' ₽';
    }

    public function getFormattedMaxDepositAttribute()
    {
        return number_format($this->max_deposit, 0) . ' ₽';
    }

    public function getFormattedMinWithdrawalAttribute()
    {
        return number_format($this->min_withdrawal, 0) . ' ₽';
    }

    public function getFormattedMaxWithdrawalAttribute()
    {
        return number_format($this->max_withdrawal, 0) . ' ₽';
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    public function scopeTopRated($query)
    {
        return $query->orderBy('rating', 'desc');
    }

    public function getRouteKeyName()
    {
        return 'slug';
    }
} 