<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\UserPoints;
use App\Models\UserFollow;
use App\Models\UserBlacklist;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }

    /**
     * Display the authenticated user's profile
     */
    public function profile()
    {
        $user = Auth::user();

        // Загружаем связанные данные
        $user->load([
            'referrals:id,name,username,created_at',
            'predictions' => function ($query) {
                $query->orderBy('match_date', 'desc')->limit(5);
            },
            'pointsHistory' => function ($query) {
                $query->orderBy('created_at', 'desc')->limit(10);
            },
            'promoCodeUsages' => function ($query) {
                $query->with(['promoCode.bookmaker'])->orderBy('created_at', 'desc')->limit(10);
            }
        ]);

        // Получаем статистику
        $stats = [
            'total_points' => $user->total_points ?? 0,
            'promo_codes_used' => $user->promoCodeUsages()->count(),
            'total_predictions' => $user->predictions()->count(),
            'referrals_count' => $user->referrals()->count(),
            'followers' => $user->followers()->count(),
            'following' => $user->follows()->count()
        ];

        $userData = $user->toArray();
        $userData['avatar_url'] = $user->getAvatarUrl();
        $userData['rank'] = $user->current_rank;
        
        // Форматируем дату рождения для корректного отображения в input type="date"
        if ($user->birth_date) {
            $userData['birth_date'] = $user->birth_date->format('Y-m-d');
        }

        return response()->json([
            'success' => true,
            'data' => [
                'user' => $userData,
                'stats' => $stats,
                'predictions' => $user->predictions->toArray(),
                'referrals' => $user->referrals->toArray(),
                'pointsHistory' => $user->pointsHistory->toArray()
            ]
        ]);
    }

    /**
     * Update the authenticated user's profile
     */
    public function updateProfile(Request $request)
    {
        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|nullable|string|max:255',
            'username' => 'sometimes|nullable|string|max:255|unique:users,username,' . $user->id,
            'email' => 'sometimes|nullable|email|unique:users,email,' . $user->id,
            'bio' => 'sometimes|nullable|string|max:1000',
            'phone' => 'sometimes|nullable|string|max:20',
            'birth_date' => 'sometimes|nullable|date',
            'gender' => 'sometimes|nullable|in:male,female,other',
            'country' => 'sometimes|nullable|string|max:100',
            'city' => 'sometimes|nullable|string|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $updateData = $request->only([
            'name', 'username', 'email', 'bio', 'phone', 'birth_date', 'gender', 'country', 'city'
        ]);
        
        foreach ($updateData as $key => $value) {
            if ($value === '') {
                $updateData[$key] = null;
            }
        }
        
        $profileCompletionRewards = $this->checkProfileCompletion($user, $updateData);
        
        $user->update($updateData);

        foreach ($profileCompletionRewards as $reward) {
            UserPoints::addPoints(
                $user->id,
                $reward['points'],
                $reward['action_type'],
                $reward['description'],
                $user
            );
        }

        $user = $user->fresh();

        $userData = $user->toArray();
        $userData['avatar_url'] = $user->getAvatarUrl();
        
        // Форматируем дату рождения для корректного отображения в input type="date"
        if ($user->birth_date) {
            $userData['birth_date'] = $user->birth_date->format('Y-m-d');
        }

        return response()->json([
            'success' => true,
            'message' => 'Profile updated successfully',
            'data' => [
                'user' => $userData
            ]
        ]);
    }

    /**
     * Get user's points history
     */
    public function pointsHistory(Request $request)
    {
        $user = Auth::user();

        $pointsHistory = UserPoints::where('user_id', $user->id)
            ->with('pointable')
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 20));

        return response()->json([
            'success' => true,
            'data' => [
                'points_history' => $pointsHistory,
                'total_points' => $user->total_points,
                'available_points' => $user->available_points
            ]
        ]);
    }

    /**
     * Get user's predictions
     */
    public function predictions(Request $request)
    {
        $user = Auth::user();

        $predictions = $user->predictions()
            ->orderBy('match_date', 'desc')
            ->paginate($request->get('per_page', 15));

        return response()->json([
            'success' => true,
            'data' => [
                'predictions' => $predictions,
                'stats' => [
                    'total_predictions' => $user->predictions_count,
                    'correct_predictions' => $user->correct_predictions,
                    'success_rate' => $user->success_rate,
                    'total_profit' => $user->total_profit
                ]
            ]
        ]);
    }

    /**
     * Get user's referrals
     */
    public function referrals(Request $request)
    {
        $user = Auth::user();

        $referrals = $user->referrals()
            ->select('id', 'name', 'username', 'email', 'created_at')
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 15));

        return response()->json([
            'success' => true,
            'data' => [
                'referrals' => $referrals,
                'stats' => [
                    'total_referrals' => $user->referrals()->count(),
                    'referral_code' => $user->referral_code
                ]
            ]
        ]);
    }

    /**
     * Get user's promo code usage history
     */
    public function promoCodeHistory(Request $request)
    {
        $user = Auth::user();

        $usageHistory = $user->promoCodeUsages()
            ->with(['promoCode.bookmaker'])
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 15));

        return response()->json([
            'success' => true,
            'data' => [
                'usage_history' => $usageHistory
            ]
        ]);
    }

    /**
     * Get user statistics
     */
    public function statistics()
    {
        $user = Auth::user();

        $stats = [
            'total_points' => $user->total_points,
            'available_points' => $user->available_points,
            'total_profit' => $user->total_profit,
            'predictions_count' => $user->predictions_count,
            'correct_predictions' => $user->correct_predictions,
            'success_rate' => $user->success_rate,
            'referrals_count' => $user->referrals()->count(),
            'promo_codes_used' => $user->promoCodeUsages()->count(),
            'member_since' => $user->created_at->diffForHumans(),
            'last_login' => $user->last_login_at ? $user->last_login_at->diffForHumans() : 'Never'
        ];

        return response()->json([
            'success' => true,
            'data' => [
                'statistics' => $stats
            ]
        ]);
    }

    /**
     * Get leaderboard
     */
    public function leaderboard(Request $request)
    {
        $type = $request->get('type', 'points');
        $limit = $request->get('limit', 20);

        switch ($type) {
            case 'success_rate':
                $users = User::where('predictions_count', '>=', 5)
                    ->where('success_rate', '>', 0)
                    ->orderBy('success_rate', 'desc')
                    ->orderBy('correct_predictions', 'desc')
                    ->limit($limit)
                    ->get(['id', 'name', 'username', 'avatar', 'success_rate', 'predictions_count', 'correct_predictions', 'rank_id', 'current_points']);
                break;
            case 'profit':
                $users = User::where('total_profit', '>', 0)
                    ->orderBy('total_profit', 'desc')
                    ->limit($limit)
                    ->get(['id', 'name', 'username', 'avatar', 'total_profit', 'predictions_count', 'rank_id', 'current_points']);
                break;
            case 'referrals':
                $users = User::whereHas('referrals')
                    ->withCount('referrals')
                    ->orderBy('referrals_count', 'desc')
                    ->limit($limit)
                    ->get(['id', 'name', 'username', 'avatar', 'referrals_count', 'rank_id', 'current_points']);
                break;
            default: // points
                $users = User::where('total_points', '>', 0)
                    ->orderBy('total_points', 'desc')
                    ->limit($limit)
                    ->get(['id', 'name', 'username', 'avatar', 'total_points', 'available_points', 'rank_id', 'current_points']);
        }

        // Add avatar_url and rank to users
        $usersData = $users->map(function ($user) {
            $userData = $user->toArray();
            $userData['avatar_url'] = $user->getAvatarUrl();
            $userData['rank'] = $user->current_rank;
            return $userData;
        })->toArray();

        return response()->json([
            'success' => true,
            'data' => [
                'leaderboard' => $usersData,
                'type' => $type
            ]
        ]);
    }

    /**
     * Get online users
     */
    public function getOnlineUsers(Request $request)
    {
        $limit = $request->get('limit', 20);
        
        $onlineUsers = User::where('is_online', true)
            ->where('last_activity_at', '>=', now()->subMinutes(5))
            ->select([
                'id', 'name', 'username', 'avatar', 'role', 
                'last_activity_at', 'is_online'
            ])
            ->orderBy('last_activity_at', 'desc')
            ->limit($limit)
            ->get()
            ->map(function ($user) {
                return [
                    'id' => $user->id,
                    'name' => $user->name,
                    'username' => $user->username,
                    'avatar' => $user->avatar,
                    'avatar_url' => $user->getAvatarUrl(),
                    'role' => $user->role,
                    'online_status' => [
                        'is_online' => $user->isCurrentlyOnline(),
                        'status_text' => $user->getOnlineStatusText(),
                        'status_class' => $user->getOnlineStatusClass(),
                        'last_activity' => $user->last_activity_at ? $user->last_activity_at->diffForHumans() : null
                    ]
                ];
            });

        return response()->json([
            'success' => true,
            'data' => [
                'online_users' => $onlineUsers,
                'total_online' => $onlineUsers->count()
            ]
        ]);
    }

    /**
     * Get user by username
     */
    public function showByUsername($username)
    {
        // Логируем входящий параметр для отладки
        \Log::info('showByUsername called with username: ' . $username);
        \Log::info('Raw username: ' . urldecode($username));
        
        $user = User::where('username', $username)
            ->select([
                'id', 'name', 'username', 'email', 'avatar', 'bio', 'phone', 
                'birth_date', 'gender', 'country', 'city', 'role', 'referral_code',
                'total_points', 'available_points', 'total_profit',
                'predictions_count', 'correct_predictions', 'success_rate', 
                'is_verified', 'last_login_at', 'last_activity_at', 'is_online', 'created_at',
                'rank_id', 'current_points'
            ])
            ->first();

        if (!$user) {
            \Log::warning('User not found for username: ' . $username);
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
        }

        \Log::info('User found: ' . $user->username);

        // Добавляем поле location, объединяя country и city
        $user->location = trim(implode(', ', array_filter([$user->country, $user->city])));
        
        // Добавляем поле is_active на основе is_verified
        $user->is_active = $user->is_verified;

        // Добавляем информацию об онлайн статусе
        $user->online_status = [
            'is_online' => $user->isCurrentlyOnline(),
            'status_text' => $user->getOnlineStatusText(),
            'status_class' => $user->getOnlineStatusClass(),
            'last_activity' => $user->last_activity_at ? $user->last_activity_at->diffForHumans() : null
        ];

        // Подготавливаем данные пользователя с avatar_url и rank
        $userData = $user->toArray();
        $userData['avatar_url'] = $user->getAvatarUrl();
        $userData['rank'] = $user->current_rank;
        
        // Форматируем дату рождения для корректного отображения в input type="date"
        if ($user->birth_date) {
            $userData['birth_date'] = $user->birth_date->format('Y-m-d');
        }

        return response()->json([
            'success' => true,
            'data' => $userData
        ]);
    }

    /**
     * Upload user avatar
     */
    public function uploadAvatar(Request $request)
    {
        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'avatar' => 'required|image|mimes:jpeg,png,jpg,gif,webp,svg,bmp,tiff,ico|max:2048'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Удаляем старый аватар если есть
            if ($user->avatar) {
                \Storage::disk('public')->delete($user->avatar);
            }

            // Сохраняем новый аватар
            $path = $request->file('avatar')->store('avatars', 'public');
            
            // Проверяем, была ли это первая загрузка аватара
            $isFirstUpload = !$user->avatar;
            
            $user->update(['avatar' => $path]);

            // Начисляем очки за первую загрузку аватара
            if ($isFirstUpload) {
                UserPoints::addPoints(
                    $user->id,
                    25,
                    'avatar_uploaded',
                    'Загрузка аватара',
                    $user
                );
            }

            // Обновляем пользователя для получения свежих данных
            $user = $user->fresh();

            return response()->json([
                'success' => true,
                'message' => 'Avatar uploaded successfully',
                'data' => [
                    'avatar_url' => $user->getAvatarUrl()
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error uploading avatar: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove user avatar
     */
    public function removeAvatar()
    {
        $user = Auth::user();

        try {
            if ($user->avatar) {
                \Storage::disk('public')->delete($user->avatar);
                $user->update(['avatar' => null]);
            }

            // Обновляем пользователя для получения свежих данных
            $user = $user->fresh();

            return response()->json([
                'success' => true,
                'message' => 'Avatar removed successfully',
                'data' => [
                    'avatar_url' => null
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error removing avatar: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update user's online status
     */
    public function updateOnlineStatus(Request $request)
    {
        $user = Auth::user();
        
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $user->updateOnlineStatus(true);

        return response()->json([
            'success' => true,
            'message' => 'Online status updated',
            'data' => [
                'is_online' => $user->isCurrentlyOnline(),
                'status_text' => $user->getOnlineStatusText(),
                'status_class' => $user->getOnlineStatusClass(),
                'last_activity' => $user->last_activity_at ? $user->last_activity_at->diffForHumans() : null
            ]
        ]);
    }

    /**
     * Get predictors list
     */
    public function getPredictors(Request $request)
    {
        $query = User::query();
        
        // Исключаем заблокированных пользователей
        if (Auth::check()) {
            $user = Auth::user();
            $blockedUserIds = $user->blockedUsers()->pluck('blocked_user_id');
            $blockedByUserIds = $user->blockedByUsers()->pluck('user_id');
            
            $query->whereNotIn('id', $blockedUserIds)
                  ->whereNotIn('id', $blockedByUserIds);
        }
        
        // Search by name or username
        if ($request->search) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('username', 'like', "%{$search}%");
            });
        }
        
        // Filter by success rate
        if ($request->success_rate) {
            $rates = is_array($request->success_rate) ? $request->success_rate : [$request->success_rate];
            $query->where(function ($q) use ($rates) {
                foreach ($rates as $rate) {
                    $q->orWhere('success_rate', '>=', $rate);
                }
            });
        }
        
        // Filter by experience (predictions count)
        if ($request->experience) {
            $counts = is_array($request->experience) ? $request->experience : [$request->experience];
            $query->where(function ($q) use ($counts) {
                foreach ($counts as $count) {
                    $q->orWhere('predictions_count', '>=', $count);
                }
            });
        }
        
        // Filter by status
        if ($request->status) {
            $statuses = is_array($request->status) ? $request->status : [$request->status];
            foreach ($statuses as $status) {
                switch ($status) {
                    case 'online':
                        $query->where('is_online', true)
                              ->where('last_activity_at', '>=', now()->subMinutes(5));
                        break;
                    case 'verified':
                        $query->where('is_verified', true);
                        break;
                    case 'following':
                        if (Auth::check()) {
                            $followingIds = Auth::user()->follows()->pluck('followed_id');
                            $query->whereIn('id', $followingIds);
                        }
                        break;
                }
            }
        }
        
        if ($request->gender) {
            $genders = is_array($request->gender) ? $request->gender : [$request->gender];
            $query->whereIn('gender', $genders);
        }
        
        if ($request->gender) {
            $genders = is_array($request->gender) ? $request->gender : [$request->gender];
            $query->whereIn('gender', $genders);
        }
        
        switch ($request->filter) {
            case 'top':
                $query->where('success_rate', '>=', 80)
                      ->where('predictions_count', '>=', 20)
                      ->orderBy('success_rate', 'desc')
                      ->orderBy('correct_predictions', 'desc');
                break;
            case 'online':
                $query->where('is_online', true)
                      ->where('last_activity_at', '>=', now()->subMinutes(5))
                      ->orderBy('last_activity_at', 'desc');
                break;
            case 'verified':
                $query->where('is_verified', true)
                      ->orderBy('success_rate', 'desc');
                break;
            default:
                $query->orderBy('success_rate', 'desc')
                      ->orderBy('predictions_count', 'desc');
        }
        
        $predictors = $query->select([
            'id', 'name', 'username', 'avatar', 'bio', 'gender', 'is_verified',
            'predictions_count', 'correct_predictions', 'success_rate', 
            'total_points', 'created_at', 'last_activity_at', 'is_online',
            'rank_id', 'current_points'
        ])
        ->paginate($request->get('per_page', 12));
        
        $predictors->getCollection()->transform(function ($predictor) {
            $predictor->online_status = [
                'is_online' => $predictor->isCurrentlyOnline(),
                'status_text' => $predictor->getOnlineStatusText(),
                'status_class' => $predictor->getOnlineStatusClass(),
                'last_activity' => $predictor->last_activity_at ? $predictor->last_activity_at->diffForHumans() : null
            ];
            
            // Добавляем текущий ранг
            $predictor->rank = $predictor->current_rank;
            
            if (Auth::check()) {
                $predictor->is_followed = Auth::user()->follows()->where('followed_id', $predictor->id)->exists();
            } else {
                $predictor->is_followed = false;
            }
            
            return $predictor;
        });

        $predictorsData = $predictors->toArray();
        $predictorsData['data'] = $predictors->getCollection()->map(function ($predictor) {
            $predictorData = $predictor->toArray();
            $predictorData['avatar_url'] = $predictor->getAvatarUrl();
            return $predictorData;
        })->toArray();

        return response()->json([
            'success' => true,
            'data' => [
                'predictors' => $predictorsData
            ]
        ]);
    }

    /**
     * Follow a user
     */
    public function followUser($userId)
    {
        $user = Auth::user();
        $userToFollow = User::findOrFail($userId);
        
        if ($user->id === $userToFollow->id) {
            return response()->json([
                'success' => false,
                'message' => 'Нельзя подписаться на самого себя'
            ], 400);
        }
        
        // Check if already following
        if ($user->follows()->where('followed_id', $userId)->exists()) {
            return response()->json([
                'success' => false,
                'message' => 'Вы уже подписаны на этого пользователя'
            ], 400);
        }
        
        $user->follows()->create([
            'followed_id' => $userId
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'Вы подписались на пользователя'
        ]);
    }

    /**
     * Unfollow a user
     */
    public function unfollowUser($userId)
    {
        $user = Auth::user();
        
        $follow = $user->follows()->where('followed_id', $userId)->first();
        
        if (!$follow) {
            return response()->json([
                'success' => false,
                'message' => 'Вы не подписаны на этого пользователя'
            ], 400);
        }
        
        $follow->delete();
        
        return response()->json([
            'success' => true,
            'message' => 'Вы отписались от пользователя'
        ]);
    }

    /**
     * Get user's followers
     */
    public function getFollowers(Request $request)
    {
        $user = Auth::user();
        
        // Получаем ID заблокированных пользователей
        $blockedUserIds = $user->blockedUsers()->pluck('blocked_user_id');
        $blockedByUserIds = $user->blockedByUsers()->pluck('user_id');
        
        $followers = $user->followers()
            ->with('follower:id,name,username,avatar')
            ->whereNotIn('follower_id', $blockedUserIds)
            ->whereNotIn('follower_id', $blockedByUserIds)
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 20));
        
        // Add avatar_url to followers
        $followersData = $followers->toArray();
        $followersData['data'] = $followers->getCollection()->map(function ($follow) {
            $followerData = $follow->follower->toArray();
            $followerData['avatar_url'] = $follow->follower->getAvatarUrl();
            return $followerData;
        })->toArray();

        return response()->json([
            'success' => true,
            'data' => [
                'followers' => $followersData
            ]
        ]);
    }

    /**
     * Get users that user follows
     */
    public function getFollowing(Request $request)
    {
        $user = Auth::user();
        
        // Получаем ID заблокированных пользователей
        $blockedUserIds = $user->blockedUsers()->pluck('blocked_user_id');
        $blockedByUserIds = $user->blockedByUsers()->pluck('user_id');
        
        // Отладочная информация
        \Log::info('getFollowing called for user ID: ' . $user->id);
        
        $following = $user->follows()
            ->with('followed:id,name,username,avatar')
            ->whereNotIn('followed_id', $blockedUserIds)
            ->whereNotIn('followed_id', $blockedByUserIds)
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 20));
        
        \Log::info('Following query result:', $following->toArray());
        
        // Add avatar_url to following
        $followingData = $following->toArray();
        $followingData['data'] = $following->getCollection()->map(function ($follow) {
            $followedData = $follow->followed->toArray();
            $followedData['avatar_url'] = $follow->followed->getAvatarUrl();
            return $followedData;
        })->toArray();

        \Log::info('Final following data:', $followingData);

        return response()->json([
            'success' => true,
            'data' => [
                'following' => $followingData
            ]
        ]);
    }

    /**
     * Get follow status for a specific user
     */
    public function getFollowStatus($userId)
    {
        $user = Auth::user();
        $targetUser = User::findOrFail($userId);
        
        $isFollowing = UserFollow::where('follower_id', $user->id)
            ->where('followed_id', $targetUser->id)
            ->exists();
        
        return response()->json([
            'success' => true,
            'data' => [
                'is_following' => $isFollowing
            ]
        ]);
    }

    /**
     * Get follow statuses for multiple users
     */
    public function getFollowStatuses(Request $request)
    {
        $user = Auth::user();
        
        $validator = Validator::make($request->all(), [
            'predictor_ids' => 'required|array',
            'predictor_ids.*' => 'integer|exists:users,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $predictorIds = $request->predictor_ids;
        
        // Получаем все подписки текущего пользователя на указанных прогнозистов
        $follows = UserFollow::where('follower_id', $user->id)
            ->whereIn('followed_id', $predictorIds)
            ->pluck('followed_id')
            ->toArray();

        // Создаем массив статусов подписок
        $followStatuses = [];
        foreach ($predictorIds as $predictorId) {
            $followStatuses[$predictorId] = in_array($predictorId, $follows);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'follow_statuses' => $followStatuses
            ]
        ]);
    }

    /**
     * Alias for getFollowing method for route compatibility
     */
    public function following(Request $request)
    {
        return $this->getFollowing($request);
    }

    /**
     * Alias for getFollowers method for route compatibility
     */
    public function followers(Request $request)
    {
        return $this->getFollowers($request);
    }

    /**
     * Get user's follows (following and followers)
     */
    public function follows(Request $request)
    {
        $user = Auth::user();

        // Получаем ID заблокированных пользователей
        $blockedUserIds = $user->blockedUsers()->pluck('blocked_user_id');
        $blockedByUserIds = $user->blockedByUsers()->pluck('user_id');

        $following = $user->follows()
            ->with(['followed:id,name,username,bio,avatar,is_verified'])
            ->whereNotIn('followed_id', $blockedUserIds)
            ->whereNotIn('followed_id', $blockedByUserIds)
            ->orderBy('created_at', 'desc')
            ->get();

        $followers = $user->followers()
            ->with(['follower:id,name,username,bio,avatar,is_verified'])
            ->whereNotIn('follower_id', $blockedUserIds)
            ->whereNotIn('follower_id', $blockedByUserIds)
            ->orderBy('created_at', 'desc')
            ->get();

        // Получаем ID всех пользователей, на которых подписаны
        $myFollowingIds = $user->follows()->pluck('followed_id')->toArray();

        // Преобразуем данные для following
        $followingData = $following->map(function ($follow) {
            $userData = $follow->followed->toArray();
            $userData['avatar_url'] = $follow->followed->getAvatarUrl();
            return $userData;
        })->toArray();

        // Преобразуем данные для followers
        $followersData = $followers->map(function ($follow) {
            $userData = $follow->follower->toArray();
            $userData['avatar_url'] = $follow->follower->getAvatarUrl();
            return $userData;
        })->toArray();

        return response()->json([
            'success' => true,
            'data' => [
                'following' => $followingData,
                'followers' => $followersData,
                'my_following_ids' => $myFollowingIds,
                'stats' => [
                    'following_count' => $user->follows()->count(),
                    'followers_count' => $user->followers()->count()
                ]
            ]
        ]);
    }

    /**
     * Change user password
     */
    public function changePassword(Request $request)
    {
        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'current_password' => 'required|string',
            'password' => 'required|string|min:8|confirmed',
            'password_confirmation' => 'required|string'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        // Проверяем текущий пароль
        if (!Hash::check($request->current_password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Current password is incorrect',
                'errors' => [
                    'current_password' => ['Текущий пароль неверен']
                ]
            ], 422);
        }

        // Обновляем пароль
        $user->update([
            'password' => Hash::make($request->password)
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Password changed successfully'
        ]);
    }

    /**
     * Get user's rank progress
     */
    public function rankProgress()
    {
        $user = Auth::user();
        
        // Получаем текущий ранг пользователя
        $currentRank = $user->current_rank;
        
        // Получаем следующий ранг
        $nextRank = null;
        if ($currentRank) {
            $nextRank = \App\Models\Rank::where('level', '>', $currentRank->level)
                ->orderBy('level', 'asc')
                ->first();
        }
        
        // Рассчитываем прогресс
        $currentPoints = $user->total_points ?? 0;
        $pointsToNextRank = 0;
        $progress = 0;
        $isMaxRank = false;
        
        if ($currentRank && $nextRank) {
            $pointsInCurrentRank = $currentPoints - $currentRank->min_points;
            $pointsNeededForNextRank = $nextRank->min_points - $currentRank->min_points;
            $pointsToNextRank = $nextRank->min_points - $currentPoints;
            $progress = $pointsNeededForNextRank > 0 ? ($pointsInCurrentRank / $pointsNeededForNextRank) * 100 : 0;
        } elseif ($currentRank && !$nextRank) {
            // Пользователь достиг максимального ранга
            $isMaxRank = true;
            $progress = 100;
        }
        
        return response()->json([
            'success' => true,
            'data' => [
                'current_rank' => $currentRank,
                'next_rank' => $nextRank,
                'current_points' => $currentPoints,
                'points_to_next_rank' => $pointsToNextRank,
                'progress' => $progress,
                'is_max_rank' => $isMaxRank
            ]
        ]);
    }

    /**
     * Delete user account
     */
    public function deleteAccount()
    {
        $user = Auth::user();

        // Удаляем все связанные данные
        $user->predictions()->delete();
        $user->pointsHistory()->delete();
        $user->promoCodeUsages()->delete();
        $user->follows()->delete();
        $user->followers()->delete();
        $user->stories()->delete();
        $user->storyViews()->delete();
        $user->storyLikes()->delete();
        
        // Обновляем рефералов, убирая ссылку на удаляемого пользователя
        $user->referrals()->update(['referred_by' => null]);
        
        // Удаляем самого пользователя
        $user->delete();

        return response()->json([
            'success' => true,
            'message' => 'Account deleted successfully'
        ]);
    }

    /**
     * Check profile completion and return rewards
     */
    private function checkProfileCompletion(User $user, array $updateData): array
    {
        $rewards = [];

        // Проверяем заполнение основной информации
        if (isset($updateData['name']) && !$user->name && $updateData['name']) {
            $rewards[] = [
                'points' => 10,
                'action_type' => 'profile_name_added',
                'description' => 'Добавление имени в профиль'
            ];
        }

        if (isset($updateData['bio']) && !$user->bio && $updateData['bio']) {
            $rewards[] = [
                'points' => 15,
                'action_type' => 'profile_bio_added',
                'description' => 'Добавление описания в профиль'
            ];
        }

        if (isset($updateData['phone']) && !$user->phone && $updateData['phone']) {
            $rewards[] = [
                'points' => 10,
                'action_type' => 'profile_phone_added',
                'description' => 'Добавление телефона в профиль'
            ];
        }

        if (isset($updateData['birth_date']) && !$user->birth_date && $updateData['birth_date']) {
            $rewards[] = [
                'points' => 10,
                'action_type' => 'profile_birthdate_added',
                'description' => 'Добавление даты рождения в профиль'
            ];
        }

        if (isset($updateData['gender']) && !$user->gender && $updateData['gender']) {
            $rewards[] = [
                'points' => 5,
                'action_type' => 'profile_gender_added',
                'description' => 'Указание пола в профиле'
            ];
        }

        if (isset($updateData['country']) && !$user->country && $updateData['country']) {
            $rewards[] = [
                'points' => 8,
                'action_type' => 'profile_country_added',
                'description' => 'Добавление страны в профиль'
            ];
        }

        if (isset($updateData['city']) && !$user->city && $updateData['city']) {
            $rewards[] = [
                'points' => 8,
                'action_type' => 'profile_city_added',
                'description' => 'Добавление города в профиль'
            ];
        }

        // Проверяем полное заполнение профиля
        $updatedUser = (object) array_merge($user->toArray(), $updateData);
        if ($this->isProfileComplete($updatedUser) && !$this->isProfileComplete($user)) {
            $rewards[] = [
                'points' => 50,
                'action_type' => 'profile_completed',
                'description' => 'Полное заполнение профиля'
            ];
        }

        return $rewards;
    }

    /**
     * Check if profile is complete
     */
    private function isProfileComplete($user): bool
    {
        $requiredFields = ['name', 'bio', 'phone', 'birth_date', 'gender', 'country', 'city'];
        
        foreach ($requiredFields as $field) {
            if (empty($user->$field)) {
                return false;
            }
        }
        
        // Также проверяем наличие аватара
        return !empty($user->avatar);
    }
}
