<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Slot;
use Illuminate\Http\Request;

class SlotController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Slot::active();

        // Фильтрация по провайдеру
        if ($request->has('provider')) {
            $query->byProvider($request->provider);
        }

        // Фильтрация по теме
        if ($request->has('theme')) {
            $query->byTheme($request->theme);
        }

        // Сортировка
        $sort = $request->get('sort', 'rating');
        switch ($sort) {
            case 'popular':
                $query->popular();
                break;
            case 'newest':
                $query->orderBy('created_at', 'desc');
                break;
            case 'rating':
            default:
                $query->topRated();
                break;
        }

        $slots = $query->paginate(12);

        return response()->json([
            'success' => true,
            'data' => [
                'slots' => $slots->items(),
                'pagination' => [
                    'current_page' => $slots->currentPage(),
                    'last_page' => $slots->lastPage(),
                    'per_page' => $slots->perPage(),
                    'total' => $slots->total(),
                ]
            ]
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show($slug)
    {
        $slot = Slot::where('slug', $slug)->active()->first();

        if (!$slot) {
            return response()->json([
                'success' => false,
                'message' => 'Слот не найден'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'slot' => $slot
            ]
        ]);
    }

    /**
     * Get featured slots.
     */
    public function featured()
    {
        $slots = Slot::active()->featured()->topRated()->limit(6)->get();

        return response()->json([
            'success' => true,
            'data' => [
                'slots' => $slots
            ]
        ]);
    }

    /**
     * Get popular slots.
     */
    public function popular()
    {
        $slots = Slot::active()->popular()->limit(6)->get();

        return response()->json([
            'success' => true,
            'data' => [
                'slots' => $slots
            ]
        ]);
    }
} 