<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Rank;
use Illuminate\Http\JsonResponse;

class RankController extends Controller
{
    /**
     * Получить список всех рангов
     */
    public function index(): JsonResponse
    {
        $ranks = Rank::orderBy('level')->get();
        
        return response()->json([
            'success' => true,
            'data' => $ranks
        ]);
    }

    /**
     * Получить конкретный ранг
     */
    public function show(string $slug): JsonResponse
    {
        $rank = Rank::where('slug', $slug)->first();
        
        if (!$rank) {
            return response()->json([
                'success' => false,
                'message' => 'Ранг не найден'
            ], 404);
        }
        
        return response()->json([
            'success' => true,
            'data' => $rank
        ]);
    }

    /**
     * Получить ранг по количеству очков
     */
    public function getByPoints(Request $request): JsonResponse
    {
        $request->validate([
            'points' => 'required|integer|min:0'
        ]);

        $points = $request->input('points');
        $rank = Rank::getRankByPoints($points);
        
        if (!$rank) {
            return response()->json([
                'success' => false,
                'message' => 'Ранг не найден для указанного количества очков'
            ], 404);
        }
        
        return response()->json([
            'success' => true,
            'data' => $rank
        ]);
    }

    /**
     * Получить прогресс пользователя до следующего ранга
     */
    public function getUserProgress(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user->rank) {
            return response()->json([
                'success' => false,
                'message' => 'У пользователя нет ранга'
            ], 404);
        }
        
        $progress = $user->getRankProgress();
        $pointsToNext = $user->getPointsToNextRank();
        $nextRank = $user->rank->getNextRank();
        
        return response()->json([
            'success' => true,
            'data' => [
                'current_rank' => $user->rank,
                'next_rank' => $nextRank,
                'progress' => round($progress, 2),
                'points_to_next_rank' => $pointsToNext,
                'current_points' => $user->current_points,
                'is_max_rank' => $user->isMaxRank()
            ]
        ]);
    }
}
