<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Jobs\ParseContentJob;
use App\Models\ParserJob;
use App\Models\ParsedPost;
use App\Services\ParserService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;

class ParserController extends Controller
{
    protected $parserService;

    public function __construct(ParserService $parserService)
    {
        $this->parserService = $parserService;
    }

    /**
     * Получить список задач парсинга
     */
    public function index(): JsonResponse
    {
        $jobs = ParserJob::withCount('parsedPosts')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return response()->json([
            'success' => true,
            'data' => $jobs
        ]);
    }

    /**
     * Создать новую задачу парсинга
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'base_url' => 'required|url',
            'pagination_pattern' => 'nullable|string|max:255',
            'start_page' => 'required|integer|min:1',
            'end_page' => 'required|integer|min:1|gte:start_page',
            'post_selector' => 'required|string|max:255',
            'title_selector' => 'required|string|max:255',
            'content_selector' => 'required|string|max:255',
            'image_selector' => 'nullable|string|max:255',
            'link_selector' => 'nullable|string|max:255',
            'settings' => 'nullable|array'
        ]);

        $job = ParserJob::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'Задача парсинга создана',
            'data' => $job
        ], 201);
    }

    /**
     * Получить задачу парсинга
     */
    public function show(ParserJob $parserJob): JsonResponse
    {
        $parserJob->load(['parsedPosts' => function ($query) {
            $query->orderBy('created_at', 'desc');
        }]);

        return response()->json([
            'success' => true,
            'data' => $parserJob
        ]);
    }

    /**
     * Обновить задачу парсинга
     */
    public function update(Request $request, ParserJob $parserJob): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'base_url' => 'sometimes|url',
            'pagination_pattern' => 'nullable|string|max:255',
            'start_page' => 'sometimes|integer|min:1',
            'end_page' => 'sometimes|integer|min:1|gte:start_page',
            'post_selector' => 'sometimes|string|max:255',
            'title_selector' => 'sometimes|string|max:255',
            'content_selector' => 'sometimes|string|max:255',
            'image_selector' => 'nullable|string|max:255',
            'link_selector' => 'nullable|string|max:255',
            'settings' => 'nullable|array'
        ]);

        $parserJob->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Задача парсинга обновлена',
            'data' => $parserJob
        ]);
    }

    /**
     * Удалить задачу парсинга
     */
    public function destroy(ParserJob $parserJob): JsonResponse
    {
        $parserJob->delete();

        return response()->json([
            'success' => true,
            'message' => 'Задача парсинга удалена'
        ]);
    }

    /**
     * Запустить парсинг
     */
    public function start(ParserJob $parserJob): JsonResponse
    {
        if ($parserJob->isRunning()) {
            return response()->json([
                'success' => false,
                'message' => 'Парсинг уже выполняется'
            ], 400);
        }

        // Сбрасываем статус
        $parserJob->update([
            'status' => 'pending',
            'error_message' => null,
            'total_posts' => 0,
            'parsed_posts' => 0
        ]);

        // Запускаем в очереди
        ParseContentJob::dispatch($parserJob);

        return response()->json([
            'success' => true,
            'message' => 'Парсинг запущен'
        ]);
    }

    /**
     * Остановить парсинг
     */
    public function stop(ParserJob $parserJob): JsonResponse
    {
        if (!$parserJob->isRunning()) {
            return response()->json([
                'success' => false,
                'message' => 'Парсинг не выполняется'
            ], 400);
        }

        $parserJob->update([
            'status' => 'stopped',
            'error_message' => 'Остановлен пользователем'
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Парсинг остановлен'
        ]);
    }

    /**
     * Получить прогресс парсинга
     */
    public function progress(ParserJob $parserJob): JsonResponse
    {
        return response()->json([
            'success' => true,
            'data' => [
                'status' => $parserJob->status,
                'progress' => $parserJob->progress,
                'total_posts' => $parserJob->total_posts,
                'parsed_posts' => $parserJob->parsed_posts,
                'error_message' => $parserJob->error_message
            ]
        ]);
    }

    /**
     * Тестировать селекторы
     */
    public function testSelectors(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'url' => 'required|url',
            'selectors' => 'required|array',
            'selectors.*' => 'string'
        ]);

        $results = $this->parserService->testSelectors(
            $validated['url'],
            $validated['selectors']
        );

        return response()->json([
            'success' => true,
            'data' => $results
        ]);
    }

    /**
     * Получить спарсенные посты
     */
    public function posts(ParserJob $parserJob): JsonResponse
    {
        $posts = $parserJob->parsedPosts()
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $posts
        ]);
    }

    /**
     * Экспортировать посты в JSON
     */
    public function export(ParserJob $parserJob): JsonResponse
    {
        $posts = $parserJob->parsedPosts()
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'job' => $parserJob->only(['id', 'name', 'base_url', 'created_at']),
                'posts' => $posts,
                'exported_at' => now()->toISOString()
            ]
        ]);
    }
}
