<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Conversation;
use App\Models\Message;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class MessengerController extends Controller
{
    /**
     * Получить список диалогов пользователя
     */
    public function conversations(Request $request): JsonResponse
    {
        $user = auth('sanctum')->user();
        
        // Детальная отладочная информация
        \Log::info('💬 Messenger conversations request', [
            'user_id' => $user ? $user->id : 'null',
            'user_name' => $user ? $user->name : 'null',
            'user_authenticated' => $user ? 'yes' : 'no',
            'request_headers' => $request->header(),
            'request_url' => $request->fullUrl(),
            'request_method' => $request->method(),
            'sanctum_guard' => auth('sanctum')->check() ? 'authenticated' : 'not_authenticated',
            'bearer_token' => $request->bearerToken(),
            'timestamp' => now()->toISOString()
        ]);
        
        $conversations = Conversation::where('user1_id', $user->id)
            ->orWhere('user2_id', $user->id)
            ->with(['userOne:id,name,username,avatar,is_online,last_activity_at', 'userTwo:id,name,username,avatar,is_online,last_activity_at', 'lastMessage'])
            ->orderByDesc('updated_at')
            ->get();
            
        \Log::info('📋 Found conversations', [
            'count' => $conversations->count(),
            'conversations' => $conversations->toArray(),
            'timestamp' => now()->toISOString()
        ]);

        // Добавляем собеседника и аватар
        $conversations->transform(function ($conv) use ($user) {
            $conv->participant = $conv->getInterlocutorSafe($user);
            if ($conv->participant) {
                $conv->participant->avatar_url = $conv->participant->getAvatarUrl();
                // Убеждаемся, что online_status всегда присутствует
                if (!isset($conv->participant->online_status)) {
                    $conv->participant->online_status = [
                        'is_online' => $conv->participant->isCurrentlyOnline(),
                        'status_text' => $conv->participant->getOnlineStatusText(),
                        'status_class' => $conv->participant->getOnlineStatusClass(),
                        'last_activity' => $conv->participant->last_activity_at ? $conv->participant->last_activity_at->diffForHumans() : null
                    ];
                }
            }
            // Добавляем количество непрочитанных сообщений для текущего пользователя
            $conv->unread_count = $conv->messages()->where('receiver_id', $user->id)->where('is_read', false)->count();
            return $conv;
        });

        $conversations = $conversations->filter(function ($conv) {
            return $conv->participant !== null;
        })->values();

        $response = [
            'success' => true,
            'data' => [
                'conversations' => $conversations
            ]
        ];
        
        \Log::info('📤 Sending conversations response', [
            'response' => $response,
            'timestamp' => now()->toISOString()
        ]);
        
        return response()->json($response);
    }

    /**
     * Получить сообщения в диалоге
     */
    public function messages($conversationId, Request $request): JsonResponse
    {
        $user = auth('sanctum')->user();
        
        \Log::info('📨 Messenger messages request', [
            'conversation_id' => $conversationId,
            'user_id' => $user ? $user->id : 'null',
            'user_authenticated' => $user ? 'yes' : 'no',
            'request_url' => $request->fullUrl(),
            'bearer_token' => $request->bearerToken(),
            'timestamp' => now()->toISOString()
        ]);
        
        if (!$user) {
            \Log::warning('❌ Messages: User not authenticated');
            return response()->json(['success' => false, 'message' => 'Необходима авторизация'], 401);
        }
        
        $conversation = Conversation::findOrFail($conversationId);
        
        \Log::info('🔍 Conversation found', [
            'conversation_id' => $conversationId,
            'user1_id' => $conversation->user1_id,
            'user2_id' => $conversation->user2_id,
            'current_user_id' => $user->id,
            'has_access' => ($conversation->user1_id == $user->id || $conversation->user2_id == $user->id)
        ]);
        
        if ($conversation->user1_id != $user->id && $conversation->user2_id != $user->id) {
            \Log::warning('❌ Messages: Access denied', [
                'conversation_id' => $conversationId,
                'user_id' => $user->id,
                'conversation_user1' => $conversation->user1_id,
                'conversation_user2' => $conversation->user2_id
            ]);
            return response()->json(['success' => false, 'message' => 'Нет доступа'], 403);
        }
        $messages = $conversation->messages()->with(['sender:id,name,username,avatar,is_online,last_activity_at', 'receiver:id,name,username,avatar,is_online,last_activity_at'])
            ->orderBy('created_at')->get();
        $messages->transform(function ($msg) {
            $msg->sender->avatar_url = $msg->sender->getAvatarUrl();
            $msg->receiver->avatar_url = $msg->receiver->getAvatarUrl();
            return $msg;
        });
        return response()->json([
            'success' => true,
            'data' => [
                'messages' => $messages
            ]
        ]);
    }

    /**
     * Отправить сообщение в диалог
     */
    public function sendMessage($conversationId, Request $request): JsonResponse
    {
        $user = auth('sanctum')->user();
        $conversation = Conversation::findOrFail($conversationId);
        if ($conversation->user1_id != $user->id && $conversation->user2_id != $user->id) {
            return response()->json(['success' => false, 'message' => 'Нет доступа'], 403);
        }
        $request->validate([
            'content' => 'required|string|max:2000',
        ]);
        // Определяем собеседника
        $otherUserId = $conversation->user1_id === $user->id ? $conversation->user2_id : $conversation->user1_id;
        $message = Message::create([
            'conversation_id' => $conversation->id,
            'sender_id' => $user->id,
            'receiver_id' => $otherUserId,
            'content' => $request->content,
            'is_read' => false,
        ]);
        $conversation->last_message_id = $message->id;
        $conversation->updated_at = now();
        $conversation->save();
        $message->load(['sender:id,name,username,avatar,is_online,last_activity_at', 'receiver:id,name,username,avatar,is_online,last_activity_at']);
        $message->sender->avatar_url = $message->sender->getAvatarUrl();
        $message->receiver->avatar_url = $message->receiver->getAvatarUrl();
        return response()->json([
            'success' => true,
            'data' => [
                'message' => $message
            ]
        ], 201);
    }

    /**
     * Начать новый диалог
     */
    public function startConversation(Request $request, $userId): JsonResponse
    {
        $user = auth('sanctum')->user();
        
        \Log::info('Starting conversation', [
            'current_user_id' => $user->id,
            'target_user_id' => $userId
        ]);
        
        // Валидируем userId из URL параметра
        $validator = \Validator::make(['user_id' => $userId], [
            'user_id' => 'required|integer|exists:users,id',
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Некорректный ID пользователя',
                'errors' => $validator->errors()
            ], 422);
        }
        
        $otherUserId = (int)$userId;
        
        if ($otherUserId == $user->id) {
            return response()->json(['success' => false, 'message' => 'Нельзя начать диалог с собой'], 400);
        }
        
        // Проверяем, существует ли уже диалог
        $conversation = Conversation::where(function ($q) use ($user, $otherUserId) {
            $q->where('user1_id', $user->id)->where('user2_id', $otherUserId);
        })->orWhere(function ($q) use ($user, $otherUserId) {
            $q->where('user1_id', $otherUserId)->where('user2_id', $user->id);
        })->first();
        
        if (!$conversation) {
            $conversation = Conversation::create([
                'user1_id' => $user->id,
                'user2_id' => $otherUserId,
            ]);
        }
        
        // Загружаем связанные данные для ответа
        $conversation->load(['userOne:id,name,username,avatar,is_online,last_activity_at', 'userTwo:id,name,username,avatar,is_online,last_activity_at']);
        
        // Добавляем участника
        $conversation->participant = $conversation->getInterlocutorSafe($user);
        if ($conversation->participant) {
            $conversation->participant->avatar_url = $conversation->participant->getAvatarUrl();
        }
        
        return response()->json([
            'success' => true,
            'data' => [
                'conversation' => $conversation
            ]
        ], 201);
    }

    /**
     * Отметить все сообщения как прочитанные
     */
    public function markAsRead($conversationId): JsonResponse
    {
        $user = auth('sanctum')->user();
        
        \Log::info('✅ Messenger markAsRead called', [
            'conversation_id' => $conversationId,
            'user_id' => $user ? $user->id : 'null',
            'user_authenticated' => $user ? 'yes' : 'no',
            'bearer_token' => request()->bearerToken(),
            'timestamp' => now()->toISOString()
        ]);
        
        if (!$user) {
            \Log::warning('❌ MarkAsRead: User not authenticated');
            return response()->json(['success' => false, 'message' => 'Необходима авторизация'], 401);
        }
        
        $conversation = Conversation::findOrFail($conversationId);
        
        \Log::info('🔍 MarkAsRead: Conversation found', [
            'conversation_id' => $conversationId,
            'user1_id' => $conversation->user1_id,
            'user2_id' => $conversation->user2_id,
            'current_user_id' => $user->id,
            'has_access' => ($conversation->user1_id === $user->id || $conversation->user2_id === $user->id)
        ]);
        
        if ($conversation->user1_id != $user->id && $conversation->user2_id != $user->id) {
            \Log::warning('❌ MarkAsRead: Access denied', [
                'conversation_id' => $conversationId,
                'user_id' => $user->id,
                'conversation_user1' => $conversation->user1_id,
                'conversation_user2' => $conversation->user2_id
            ]);
            return response()->json(['success' => false, 'message' => 'Нет доступа'], 403);
        }
        // Отмечаем все входящие сообщения как прочитанные
        $conversation->messages()->where('receiver_id', $user->id)->where('is_read', false)->update(['is_read' => true]);
        return response()->json([
            'success' => true,
            'message' => 'Все сообщения отмечены как прочитанные'
        ]);
    }

    /**
     * Удалить диалог
     */
    public function deleteConversation($conversationId): JsonResponse
    {
        $user = auth('sanctum')->user();
        $conversation = Conversation::findOrFail($conversationId);
        if ($conversation->user1_id != $user->id && $conversation->user2_id != $user->id) {
            return response()->json(['success' => false, 'message' => 'Нет доступа'], 403);
        }
        $conversation->messages()->delete();
        $conversation->delete();
        return response()->json([
            'success' => true,
            'message' => 'Диалог удалён'
        ]);
    }
} 