<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Bookmaker;
use Illuminate\Http\Request;

class CategoryController extends Controller
{
    /**
     * Display a listing of categories
     */
    public function index()
    {
        $categories = Category::active()
            ->orderBy('sort_order', 'asc')
            ->withCount('bookmakers')
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'categories' => $categories
            ]
        ]);
    }

    /**
     * Display the specified category with bookmakers
     */
    public function show($slug)
    {
        $category = Category::where('slug', $slug)
            ->active()
            ->with(['bookmakers' => function ($query) {
                $query->active()->orderBy('sort_order', 'asc');
            }])
            ->first();

        if (!$category) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'category' => $category
            ]
        ]);
    }

    /**
     * Get bookmakers by category
     */
    public function bookmakers($slug, Request $request)
    {
        $category = Category::where('slug', $slug)->active()->first();

        if (!$category) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found'
            ], 404);
        }

        $query = Bookmaker::where('category_id', $category->id)
            ->active()
            ->with('category');

        // Apply filters
        if ($request->has('featured') && $request->featured) {
            $query->featured();
        }

        if ($request->has('min_rating')) {
            $query->where('rating', '>=', $request->min_rating);
        }

        if ($request->has('min_deposit')) {
            $query->where('min_deposit', '<=', $request->min_deposit);
        }

        // Apply sorting
        $sortBy = $request->get('sort_by', 'sort_order');
        $sortOrder = $request->get('sort_order', 'asc');

        switch ($sortBy) {
            case 'rating':
                $query->orderBy('rating', $sortOrder);
                break;
            case 'name':
                $query->orderBy('name', $sortOrder);
                break;
            case 'min_deposit':
                $query->orderBy('min_deposit', $sortOrder);
                break;
            default:
                $query->orderBy('sort_order', 'asc');
        }

        $bookmakers = $query->paginate($request->get('per_page', 12));

        return response()->json([
            'success' => true,
            'data' => [
                'category' => $category,
                'bookmakers' => $bookmakers
            ]
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
