<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Bookmaker;
use App\Models\Category;
use Illuminate\Http\Request;

class BookmakerController extends Controller
{
    /**
     * Display a listing of bookmakers
     */
    public function index(Request $request)
    {
        $query = Bookmaker::active()
            ->with('category')
            ->withCount(['promoCodes' => function ($q) {
                $q->active()->valid()->available();
            }]);

        // Apply filters
        if ($request->has('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        if ($request->has('category_slug')) {
            $category = Category::where('slug', $request->category_slug)->first();
            if ($category) {
                $query->where('category_id', $category->id);
            }
        }

        if ($request->has('featured') && $request->featured) {
            $query->featured();
        }

        if ($request->has('min_rating')) {
            $query->where('rating', '>=', $request->min_rating);
        }

        if ($request->has('min_deposit')) {
            $query->where('min_deposit', '<=', $request->min_deposit);
        }

        if ($request->has('currency')) {
            $query->where('currency', $request->currency);
        }

        // Apply sorting
        $sortBy = $request->get('sort_by', 'sort_order');
        $sortOrder = $request->get('sort_order', 'asc');

        switch ($sortBy) {
            case 'rating':
                $query->topRated();
                break;
            case 'name':
                $query->orderBy('name', $sortOrder);
                break;
            case 'min_deposit':
                $query->orderBy('min_deposit', $sortOrder);
                break;
            default:
                $query->orderBy('sort_order', 'asc');
        }

        $bookmakers = $query->paginate($request->get('per_page', 12));

        return response()->json([
            'success' => true,
            'data' => [
                'bookmakers' => $bookmakers
            ]
        ]);
    }

    /**
     * Display the specified bookmaker
     */
    public function show($slug)
    {
        $bookmaker = Bookmaker::where('slug', $slug)
            ->active()
            ->with(['category', 'promoCodes' => function ($query) {
                $query->active()->valid()->available();
            }, 'bonuses' => function ($query) {
                $query->active();
            }])
            ->first();

        if (!$bookmaker) {
            return response()->json([
                'success' => false,
                'message' => 'Bookmaker not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'bookmaker' => $bookmaker
            ]
        ]);
    }

    /**
     * Get promo codes for a bookmaker
     */
    public function promoCodes($slug, Request $request)
    {
        $bookmaker = Bookmaker::where('slug', $slug)->active()->first();

        if (!$bookmaker) {
            return response()->json([
                'success' => false,
                'message' => 'Bookmaker not found'
            ], 404);
        }

        $query = $bookmaker->promoCodes()
            ->active()
            ->valid()
            ->available();

        // Apply filters
        if ($request->has('featured') && $request->featured) {
            $query->featured();
        }

        if ($request->has('bonus_type')) {
            $query->where('bonus_type', $request->bonus_type);
        }

        if ($request->has('min_bonus')) {
            $query->where('bonus_amount', '>=', $request->min_bonus);
        }

        $promoCodes = $query->paginate($request->get('per_page', 10));

        return response()->json([
            'success' => true,
            'data' => [
                'bookmaker' => $bookmaker,
                'promo_codes' => $promoCodes
            ]
        ]);
    }

    /**
     * Get bonuses for a bookmaker
     */
    public function bonuses($slug, Request $request)
    {
        $bookmaker = Bookmaker::where('slug', $slug)->active()->first();

        if (!$bookmaker) {
            return response()->json([
                'success' => false,
                'message' => 'Bookmaker not found'
            ], 404);
        }

        $query = $bookmaker->bonuses()->active();

        // Apply filters
        if ($request->has('featured') && $request->featured) {
            $query->featured();
        }

        if ($request->has('bonus_type')) {
            $query->where('bonus_type', $request->bonus_type);
        }

        $bonuses = $query->paginate($request->get('per_page', 10));

        return response()->json([
            'success' => true,
            'data' => [
                'bookmaker' => $bookmaker,
                'bonuses' => $bonuses
            ]
        ]);
    }

    /**
     * Search bookmakers
     */
    public function search(Request $request)
    {
        $query = $request->get('q');

        if (!$query || strlen($query) < 2) {
            return response()->json([
                'success' => false,
                'message' => 'Search query must be at least 2 characters long'
            ], 400);
        }

        $bookmakers = Bookmaker::active()
            ->with('category')
            ->where(function ($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                  ->orWhere('description', 'like', "%{$query}%");
            })
            ->orderBy('sort_order', 'asc')
            ->paginate($request->get('per_page', 10));

        return response()->json([
            'success' => true,
            'data' => [
                'query' => $query,
                'bookmakers' => $bookmakers
            ]
        ]);
    }

    /**
     * Get featured bookmakers
     */
    public function featured()
    {
        $bookmakers = Bookmaker::active()
            ->featured()
            ->with('category')
            ->orderBy('sort_order', 'asc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'bookmakers' => $bookmakers
            ]
        ]);
    }

    /**
     * Get top rated bookmakers
     */
    public function topRated(Request $request)
    {
        $limit = $request->get('limit', 10);

        $bookmakers = Bookmaker::active()
            ->with('category')
            ->topRated()
            ->limit($limit)
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'bookmakers' => $bookmakers
            ]
        ]);
    }
}
