<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Symfony\Component\Process\Process;

class Dev extends Command
{
    protected $signature = 'dev {--port=8000}';
    protected $description = 'Запуск серверов для разработки (Laravel + Vite)';

    public function handle()
    {
        $port = $this->option('port');

        $this->info('🚀 Запуск серверов для разработки...');
        $this->info("📡 Laravel: http://localhost:{$port}");
        $this->info('⚡ Vite: http://localhost:5173');
        $this->info('');
        $this->info('💡 Нажмите Ctrl+C для остановки');
        $this->info('');

        // Проверяем Node.js
        $nodeCheck = new Process(['node', '--version']);
        $nodeCheck->run();
        
        if (!$nodeCheck->isSuccessful()) {
            $this->error('❌ Node.js не найден! Установите Node.js');
            return 1;
        }

        // Проверяем npm
        $npmCheck = new Process(['npm', '--version']);
        $npmCheck->run();
        
        if (!$npmCheck->isSuccessful()) {
            $this->error('❌ npm не найден! Установите npm');
            return 1;
        }

        $this->info('✅ Node.js и npm найдены');
        $this->info('🔄 Запуск серверов...');

        // Создаем процессы
        $laravelProcess = new Process(['php', 'artisan', 'serve', '--port=' . $port]);
        $viteProcess = new Process(['npm', 'run', 'dev']);
        
        // Настраиваем процессы
        $laravelProcess->setTimeout(null);
        $viteProcess->setTimeout(null);
        
        // Запускаем Laravel
        $laravelProcess->start();
        $this->info('✅ Laravel сервер запущен');
        
        // Ждем немного
        sleep(2);
        
        // Запускаем Vite
        $viteProcess->start();
        $this->info('✅ Vite сервер запущен');
        
        $this->info('');
        $this->info('🎉 Оба сервера запущены!');
        $this->info('🌐 Откройте: http://localhost:' . $port);
        $this->info('');
        
        // Ждем завершения процессов
        while ($laravelProcess->isRunning() || $viteProcess->isRunning()) {
            sleep(1);
            
            // Проверяем вывод Laravel
            if ($laravelProcess->isRunning()) {
                $output = $laravelProcess->getIncrementalOutput();
                if ($output) {
                    $this->line($output);
                }
            }
            
            // Проверяем вывод Vite
            if ($viteProcess->isRunning()) {
                $output = $viteProcess->getIncrementalOutput();
                if ($output) {
                    $this->line($output);
                }
            }
        }
        
        // Останавливаем процессы
        if ($laravelProcess->isRunning()) {
            $laravelProcess->stop();
        }
        if ($viteProcess->isRunning()) {
            $viteProcess->stop();
        }
        
        $this->info('👋 Серверы остановлены');
        
        return 0;
    }
} 