# Настройка WebSocket для чата

## Текущее состояние

✅ **Что уже настроено:**
- Модуль Messenger с полной функциональностью
- API endpoints для работы с сообщениями
- Event `MessageSent` для broadcasting
- Каналы broadcasting в `routes/channels.php`
- WebSocket сервер команды
- Laravel Echo и Pusher установлены
- Компонент Messages.vue с WebSocket логикой

## Что нужно сделать

### 1. Настройка .env файла

Добавьте следующие настройки в ваш `.env` файл:

```env
# Broadcasting settings
BROADCAST_DRIVER=pusher

# Локальный WebSocket сервер (без облачных сервисов)
PUSHER_APP_ID=local-app
PUSHER_APP_KEY=local-key
PUSHER_APP_SECRET=local-secret
PUSHER_HOST=127.0.0.1
PUSHER_PORT=6001
PUSHER_SCHEME=http
PUSHER_APP_CLUSTER=local

# Messenger settings
MESSENGER_ENABLED=true
MESSENGER_WEBSOCKET_ENABLED=true
MESSENGER_WEBSOCKET_HOST=127.0.0.1
MESSENGER_WEBSOCKET_PORT=6001
```

### 2. Обновление bootstrap.js

Файл `resources/js/bootstrap.js` уже обновлен с инициализацией Echo:

```javascript
import Echo from 'laravel-echo';
import Pusher from 'pusher-js';

window.Pusher = Pusher;

window.Echo = new Echo({
    broadcaster: 'pusher',
    key: 'local-key',
    cluster: 'local',
    wsHost: window.location.hostname,
    wsPort: 6001,
    forceTLS: false,
    disableStats: true,
    enabledTransports: ['ws'],
    auth: {
        headers: {
            'X-CSRF-TOKEN': token ? token.content : '',
            'Authorization': `Bearer ${localStorage.getItem('token') || ''}`
        }
    }
});
```

### 3. Обновление Messages.vue

Компонент `resources/js/components/Profile/Messages.vue` уже обновлен для использования глобального Echo.

### 4. Запуск серверов

Используйте одну из команд для запуска:

```bash
# Для всех устройств (рекомендуется)
php artisan serve:all

# Только локальный доступ
php artisan serve:local

# Только публичный доступ (требует npm run build)
php artisan serve:public
```

### 5. Проверка работы

1. Откройте `http://localhost:8000/profile/messages`
2. Откройте консоль браузера (F12)
3. Должны появиться сообщения:
   - "WebSocket initialized using global Echo"
   - "Subscribed to conversation {id}"

## Возможные проблемы и решения

### Проблема: "Global Echo not found"
**Решение:** Убедитесь, что в `bootstrap.js` правильно импортирован Echo и он доступен в `window.Echo`

### Проблема: "Connection failed"
**Решение:** 
1. Проверьте, что WebSocket сервер запущен на порту 6001
2. Проверьте настройки в `.env`
3. Убедитесь, что в `bootstrap.js` правильный `wsHost` и `wsPort`

### Проблема: "Authentication failed"
**Решение:**
1. Проверьте, что пользователь авторизован
2. Убедитесь, что токен правильно передается в заголовках
3. Проверьте настройки каналов в `routes/channels.php`

### Проблема: Сообщения не приходят в реальном времени
**Решение:**
1. Проверьте, что Event `MessageSent` правильно настроен
2. Убедитесь, что канал `messenger.conversation.{id}` правильно подписан
3. Проверьте логи Laravel на наличие ошибок

## Тестирование

Для тестирования WebSocket функциональности:

1. Откройте чат в двух разных вкладках браузера
2. Отправьте сообщение в одной вкладке
3. Сообщение должно мгновенно появиться во второй вкладке

## Дополнительные настройки

### Для продакшена

В продакшене рекомендуется использовать:
- SSL/TLS для WebSocket соединений
- Настройка `forceTLS: true`
- Использование `wss://` вместо `ws://`

### Для разработки

Для разработки можно использовать:
- Локальный WebSocket сервер
- Настройка `forceTLS: false`
- Использование `ws://` для быстрого тестирования 